Attribute VB_Name = "lcd"

' IO-Warrior library API functions
' LCD functions
' Note: these functions are intended to be used with HD44780-based LCDs,
' and have been tested only with them.

' Enable LCD interface
' Parameters:
' iowHandle - IO-Warrior handle
' numPipe - interface number
' enable - 1 to enable, 0 to disable
Public Function IowKitEnableLCD _
    (iowHandle As Long, _
     numPipe As Long, _
     enable As Byte _
    ) _
    As Long
' LCD enable packet
Dim lcdPacket(7) As Byte
Dim nWritten As Long

' Set up IIC enable packet
' Report ID 4 is LCD enable/disable
lcdPacket(0) = 4
' Enable/disable LCD
' If enable is on, LCD is powered on and backlight is turned on
lcdPacket(1) = enable
' Write Report
nWritten = IowKitWrite(iowHandle, numPipe, lcdPacket(0), 8)
' Return number of bytes written  to IO-Warrior
IowKitEnableLCD = nWritten
End Function

' Write command to LCD
' Parameters:
' iowHandle - IO-Warrior handle
' numPipe - interface number
' Cmd - Command to be sent to LCD
Public Function IowKitWriteCmdLCD _
    (iowHandle As Long, _
     numPipe As Long, _
     Cmd As Byte) _
     As Long
' LCD Report/packet
Dim lcdPacket(7) As Byte
' Number of bytes written
Dim nWritten As Long

' Set up LCD packet
' Report ID 5 - Write to LCD
lcdPacket(0) = 5
' Instruction register, 1 bytes to write
lcdPacket(1) = 1
' Save command in packet
lcdPacket(2) = Cmd
' Go ahead and write to IO-Warrior
nWritten = IowKitWrite(iowHandle, numPipe, lcdPacket(0), 8)
' Return number of bytes written to IO-Warrior
IowKitWriteCmd = nWritten
End Function

' Initialize LCD
' Parameters:
' iowHandle - IO-Warrior handle
' numPipe - Interface number
' numLines - Number of lines
Public Function IowKitInitLCD _
    (iowHandle As Long, _
     numPipe As Long, _
     numLines As Long) _
     As Long
' LCD Report/packet
Dim lcdPacket(7) As Byte
' Number of bytes written
Dim nWritten As Long

' Set up LCD packet
' Report ID 5 - Write to LCD
lcdPacket(0) = 5
' Instruction register, 3 bytes to write
lcdPacket(1) = 3
' Init LCD
lcdPacket(2) = &H28
' More than 1 line? Set that flag
If numLines >= 1 Then
        lcdPacket(2) = lcdPacket(2) Or &H10
End If
' Clear LCD screen
lcdPacket(3) = 1
' Set cursor to line and blinking and move it to home position
lcdPacket(4) = &HF
' Write to LCD
nWritten = IowKitWrite(iowHandle, numPipe, lcdPacket(0), 8)
' After this write, LCD is initialized, DDRAM/cursor address is set to zero,
' and you can write to LCD screen
IowKitInitLCD = nWritten
End Function

' Clear LCD screen
' Parameters:
' iowHandle - IO-Warrior handle
' numPipe - Interface number
Public Function IowKitClearLCD _
    (iowHandle As Long, _
     numPipe As Long) _
     As Long
' Send clear screen command to LCD
IowKitClearLCD = IowKitWriteCmdLCD(iowHandle, numPipe, 1)
End Function

' Entry mode set - set cursor move direction, shift display
' Parameters:
' iowHandle - IO-Warrior handle
' numPipe - Interface number
' moveForward - cursor moves forward or backward
' shiftDisp - display shifts on write
Public Function IowKitEntryModeSet _
    (iowHandle As Long, _
     numPipe As Long, _
     moveForward As Boolean, _
     shiftDisp As Boolean) _
     As Long

Dim Cmd As Byte

' Entry mode set command
Cmd = &H4
' Set move direction - forward
If moveForward Then
        Cmd = Cmd Or 2
End If
' Set display shift mode
If shiftDisp Then
        Cmd = Cmd Or 1
End If
' Set "entry mode set" command to LCD
IowKitEntryModeSet = IowKitWriteCmdLCD(iowHandle, numPipe, Cmd)
End Function

' Move cursor to 1st row, 1st col of LCD screen
' Parameters:
' iowHandle - IO-Warrior handle
' numPipe - Interface number
Public Function IowKitCursorHomeLCD _
    (iowHandle As Long, _
     numPipe As Long) _
     As Long
' Send cursor home command to LCD
IowKitCursorHomeLCD = IowKitWriteCmdLCD(iowHandle, numPipe, 3)
End Function

' Display control
' Parameters:
' iowHandle - IO-Warrior handle
' numPipe - Interface number
' dispOn - turn display on or off
' cursorOn - turn cursor on or off
' cursorBlink - turn blinking of cursor on or off
Public Function IowKitDispControlLCD _
    (iowHandle As Long, _
     numPipe As Long, _
     dispOn As Boolean, _
     cursonOn As Boolean, _
     cursorBlink As Boolean) _
     As Long
' Command
Dim Cmd As Byte

' Init command
Cmd = &H8
' Check and set dispon, cursonon and cursorblink bits
If dispOn Then
        Cmd = Flags Or 4
End If
If cursorOn Then
        Cmd = Flags Or 2
End If
If cursorBlink Then
        Cmd = Flags Or 1
End If
' Send display control command to LCD
IowKitDispControlLCD = IowKitWriteCmdLCD(iowHandle, numPipe, Cmd)
End Function

' Shift mode control
' Parameters:
' iowHandle - IO-Warrior handle
' numPipe - Interface number
' shiftDisp - Shift display
' shiftDir - Shift direction - left or right
Public Function IowKitShiftControlLCD _
    (iowHandle As Long, _
     numPipe As Long, _
     shiftDisp As Boolean, _
     shiftDir As Boolean) _
     As Long
' Command
Dim Cmd As Byte

' Cursor/display shift command
Cmd = &H10
' Check and set bits
If shiftDisp Then
        Cmd = Cmd Or 8
End If
If shiftDir Then
        Cmd = Cmd Or 4
End If
' Send "Shift control" command to LCD
IowKitShiftControlLCD = IowKitWriteCmdLCD(iowHandle, numPipe, Cmd)
End Function

' Write string to LCD
' Parameters:
' iowHandle - IO-Warrior handle
' numPipe - Interface number
' strLcd - String to write
' NOTE: This function can write only one line of screen
' you have to manage multiline writes by yourself
Public Function IowKitWriteStringLCD _
    (iowHandle As Long, _
     numPipe As Long, _
     strLcd As String) _
     As Long
' Local variables
Dim lcdPacket(7) As Byte
Dim nWritten As Long
Dim nChars As Long
Dim nCurr As Long
Dim nNow As Long

' Set up packet for writing
' Report ID 5 - Write to LCD
lcdPacket(0) = 5
nCurr = 0
' Get string length
nChars = Len(strLcd)
Do While nCurr < nChars
        ' Check if we write last time
        If (nCurr + 6) > nChars Then
                nNow = nChars - nCurr
        Else
                nNow = 6
        End If
        ' Flags Or number of characters to write
        lcdPacket(1) = &H80 Or nNow
        For I = 1 To nNow
            ' Save characters to packet
            lcdPacket(1 + I) = Asc(Mid(strLcd, nCurr + I, 1))
        Next I
        ' Write to LCD
        nWritten = IowKitWrite(iowHandle, numPipe, lcdPacket(0), 8)
        nCurr = nCurr + 6
Loop
' Return number of bytes written to IO-Warrior
IowKitWriteStringLCD = nWritten
End Function

' Set DDRAM address
' Parameters:
' iowHandle - IO-Warrior handle
' numPipe - Interface number
' Address - DDRAM address
Public Function IowKitSetDDRAMAddr _
    (iowHandle As Long, _
     numPipe As Long, _
     Address As Byte) _
     As Long
' Local variables
' Command
Dim Cmd As Byte

' Command - Set DDRAM address or Address
Cmd = &H80 Or Address
' Send "Set DDRAM Address" command to LCD
IowKitSetDDRAMAddr = IowKitWriteCmdLCD(iowHandle, numPipe, Cmd)
End Function

' This functions moves cursor to new position
' Parameters:
' iowHandle - IO-Warrior handle
' numPipe - Interface number
' Row - Row
' Col - Col
' Due to weird addressing modes, some 1-line models act like 2-line models,
' so you have first half of screen on first line, and second half is on
' second line (ie at 40H address)
' See your LCD module specs for more details on addressing
Public Function IowKitSetCursorLCD _
    (iowHandle As Long, _
     numPipe As Long, _
     Row As Byte, _
     Col As Byte) _
     As Long
' Local variables
Dim nWritten As Long
Dim Address As Byte

' Calculate DDRAM memory address
If Row = 1 Then
        Address = &H0
End If
If Row = 2 Then
        Address = &H40
End If
If Row = 3 Then
        Address = &H14
End If
If Row = 4 Then
        Address = &H54
End If
Address = Address + Col - 1
' Set DDRAM address and move cursor to new position
nWritten = IowKitSetDDRAMAddr(iowHandle, numPipe, Address)
' Return number of bytes written to IO-Warrior
IowKitSetCursorLCD = nWritten
End Function

'
' LCD display control functions
'

' Turn LCD display on
' Parameters:
' iowHandle - IO-Warrior handle
' numPipe - Interface number
Public Function IowKitCursorDispOnLCD _
    (iowHandle As Long, _
     numPipe As Long) _
     As Long

' Send "Display on" command
IowKitDispOnLCD = IowKitWriteCmdLCD(iowHandle, numPipe, &HC)
End Function

' Turn LCD display off
' Parameters:
' iowHandle - IO-Warrior handle
' numPipe - Interface number
Public Function IowKitDispOffLCD _
    (iowHandle As Long, _
     numPipe As Long) _
     As Long

' Send "display off" command
IowKitDispOffLCD = IowKitWriteCmdLCD(iowHandle, numPipe, &H8)
End Function

'
' LCD cursor control functions
'

' Turn LCD cursor on
' Parameters:
' iowHandle - IO-Warrior handle
' numPipe - Interface number
Public Function IowKitCursorOnLCD _
    (iowHandle As Long, _
     numPipe As Long) _
     As Long

' Send "turn cursor on" command
IowKitCursorOnLCD = IowKitWriteCmdLCD(iowHandle, numPipe, &HD)
End Function

' Turn LCD cursor off
' Parameters:
' iowHandle - IO-Warrior handle
' numPipe - Interface number
Public Function IowKitCursorOffLCD _
    (iowHandle As Long, _
     numPipe As Long) _
     As Long

' Send "turn cursor off" command
IowKitCursorOffLCD = IowKitWriteCmdLCD(iowHandle, numPipe, &HC)
End Function

'
' LCD Cursor movement functions
'

' Move LCD cursor to the left
' Parameters:
' iowHandle - IO-Warrior handle
' numPipe - Interface number
Public Function IowKitCursorLeftLCD _
    (iowHandle As Long, _
     numPipe As Long) _
     As Long

' Send "move cursor the left" command
IowKitCursorLeftLCD = IowKitWriteCmdLCD(iowHandle, numPipe, &H10)
End Function

' Move LCD cursor to the right
' Parameters:
' iowHandle - IO-Warrior handle
' numPipe - Interface number
Public Function IowKitCursorRightLCD _
    (iowHandle As Long, _
     numPipe As Long) _
     As Long

' Send "move cursor to the right" command
IowKitCursorRightLCD = IowKitWriteCmdLCD(iowHandle, numPipe, &H14)
End Function
